/*
 * ----------------------------------------------------------------------------
 *
 *  Google Calendar on your Desktop!
 *
 *  Copyright  2007, James Marsh; portions copyright  2006, Manas Tungare.
 *
 *  CHANGELOG:
 *    2006-10-23  :  Initially created by Manas Tungare
 *	  2007-1-1    :  Revised by James Marsh
 *						- Added ability to choose how many days to show
 *						- Added ability to have calendars added but not shown
 *						- Added days shown on calendar events
 *						- Fixed timezones
 *                     
 *					  Known Issues
 *						- Widget crashes if enter is pressed while on the backside
 *
 * ----------------------------------------------------------------------------
 */

// CONSTANTS ------------------------------------------------------------------

var kCurrentVersion = "2.0";

var kProductHomePageUrl =
    "http://jamesmar.marshplace.com/";

var kUpdateIntervalMs = 1000 * 60 * 10;  // refresh every 10 minutes
var kOneDay = 1000 * 60 * 60 * 24; // day constant

// colors to differentiate calendars
var kColors = new Array();
kColors[0] = '#CC3333';
kColors[1] = '#109618';
kColors[2] = '#6633CC';
kColors[3] = '#EE8800';
kColors[4] = '#DD5511';
kColors[5] = '#AAAA11';
kColors[6] = '#DD4477';
kColors[7] = '#8C6D8C';
kColors[8] = '#D6AE00';
kColors[9] = '#7083A8';
 
// GLOBALS --------------------------------------------------------------------

// Apple Buttons
var doneButton;
var infoButton;

// Main Events Array
var todaysEvents;

// Main Calendars Array
var calendars = new Array();

// Globals for adding and removing calendars
var newURL;
var newTitle;

// CODE /////////////////////////////////////////////////////////////////////////////

// Setup functions -------------------------------------------------------------------

function widget_onLoad() {
	setupScrollbars();
	setupWidgetBack();
	loadPrefs();
	drawCalendarList();
	refresh();
}

function setupScrollbars() {
  // create events scrollbar on the front
  scrollbar = new AppleVerticalScrollbar(document.getElementById("scrollbar"));
  scrollArea = new AppleScrollArea(document.getElementById("eventList"));
  scrollArea.addScrollbar(scrollbar);
  scrollArea.focus();  // for key control when first loading in Safari
}

function setupWidgetBack() {
	// create buttons
	infoButton = new AppleInfoButton(document.getElementById("infoButton"), 
	    document.getElementById("front"), "black", "black", showPrefs);
	doneButton = new AppleGlassButton(document.getElementById("doneButton"),
	    "Done", hidePrefs);
}

// Saving and Loading Preferences -------------------------------------------------------

function loadPrefs() {
	if (window.widget) {

		document.getElementById('showPastEvents').checked = 
		    (widget.preferenceForKey('ShowPastEvents') == "true"); // If ShowPastEvents check the checkbox
		document.getElementById('showTodaysEvents').checked = 
		    (widget.preferenceForKey('ShowTodaysEvents') == "true"); // If ShowTodaysEvents check the checkbox
		document.getElementById('showDays').value = 
			widget.preferenceForKey('ShowDays');  // Load the number of days to be shown

		//// Load Calendar Urls/Titles/Checkboxes ////////////////////
		// load a string for each property since widgets can only save strings
		var calendarUrlStr = (widget.preferenceForKey('CalendarUrls') == null) ? "" :
			widget.preferenceForKey('CalendarUrls');
		var calendarTitleStr = (widget.preferenceForKey('CalendarTitles') == null) ? "" :
			widget.preferenceForKey('CalendarTitles');
		var calendarActiveStr = (widget.preferenceForKey('CalendarActive') == null) ? "" :
			widget.preferenceForKey('CalendarActive');

		// create arrays
		var calendarUrlTemp = new Array();
		var calendarTitleTemp = new Array();
		var calendarActiveTemp = new Array();

		// divide strings into arrays
		calendarUrlTemp = null;
		calendarUrlTemp = calendarUrlStr.split("|");
		calendarTitleTemp = null;
		calendarTitleTemp = calendarTitleStr.split("|");
		calendarActiveTemp = null;
		calendarActiveTemp = calendarActiveStr.split("|");
		
		// clear calendars and load values into array
		calendars = new Array();
		for(var i = 0; i < calendarUrlTemp.length; i++){
			if(	calendarTitleTemp[i] != "" && calendarUrlTemp[i] != ""){ // check something is in the strings
				if(calendarActiveTemp[i] == "true"){
					var newCal = new Calendar(calendarTitleTemp[i], calendarUrlTemp[i], true);
				}
				else{
					var newCal = new Calendar(calendarTitleTemp[i], calendarUrlTemp[i], false);
				}
				calendars.push(newCal);
			} // End if(Title or Url == "")
		} // End for(Url array length)
		// End Load Calendars
	} // End if(widget.window), in widget window
}


function savePrefs() {
	if (window.widget) {

		widget.setPreferenceForKey(document.getElementById("showPastEvents").checked ? 
		    "true" : "false", "ShowPastEvents"); // If checked save ShowPastEvents as true
		widget.setPreferenceForKey(document.getElementById("showTodaysEvents").checked ? 
		    "true" : "false", "ShowTodaysEvents"); // If checked save ShowTodaysEvents as true
		widget.setPreferenceForKey(document.getElementById("showDays").value,
			"ShowDays"); // Save the number of days to show
		
		//// SAVE CALENDAR URLs //////////////////
		// clear the strings
		var calendarUrlStr = "";
		var calendarTitleStr = "";
		var calendarActiveStr = "";
				
		// iterate the calendars array and save values to strings
		for(var j = 0; j < calendars.length; j++){
			if(calendars[j].url != null && calendars[j].url.length != 0){
				calendarUrlStr = calendarUrlStr + calendars[j].url + "|";
				calendarTitleStr = calendarTitleStr + calendars[j].title + "|";
				calendarActiveStr = calendarActiveStr + (document.getElementById(calendars[j].url).checked ? "true" : "false") + "|";
			}
		}
				
		// save the strings
		widget.setPreferenceForKey(calendarUrlStr, "CalendarUrls");
		widget.setPreferenceForKey(calendarTitleStr, "CalendarTitles");	
		widget.setPreferenceForKey(calendarActiveStr, "CalendarActive");			
	}
}

// Add and Remove Calendars --------------------------------------------------------------

function addURL() 
{
	newURL = document.getElementById('calendarUrls').value;
	newTitle = null;
	
	var exists = false;
	for(var i = 0; i < calendars.length; i++)
		if(calendars[i].url == newURL)
			exists = true;
	
	// Display error codes
	if(newURL == null || newURL.length == 0)
		document.getElementById('calendarUrls').value = "Please enter a URL";
	else if(exists == true)
		document.getElementById('calendarUrls').value = "Calendar Already Exists";
	else{
		getCalendarTitle(newURL);
		if( newTitle == undefined )
			document.getElementById('calendarUrls').value = "Invalid Calendar URL";
		else{ 
			var newCal = new Calendar(newTitle, newURL, true);
			calendars.push(newCal);
			document.getElementById('calendarUrls').value = "Calendar Added";
		}
	}
	
	// Redraw the calendar table
	drawCalendarList();
	savePrefs(); // save and load to perge deleted entries
	loadPrefs();
	drawCalendarList();

}

function removeURL() {
  // search for any unchecked calendars and remove them
  for (var i = 0; i < calendars.length; i++){
    if(document.getElementById(calendars[i].url).checked == false){
		calendars[i] = new Calendar("","", false);
		document.getElementById('calendarUrls').value = "Erased";
	}
  }
  
  // Redraw the calendar table
  savePrefs(); // save and load to perge deleted entries
  loadPrefs();
  drawCalendarList();
}

// Flip functions --------------------------------------------------

function showPrefs() {

  // get preferences and prepare to load back
  loadPrefs();
  document.getElementById('calendarUrls').value = "";
	
  var front = document.getElementById("front");
  var back = document.getElementById("back");

  if (window.widget) {
    widget.prepareForTransition("ToBack");
  }
 
  front.style.display="none";
  back.style.display="block";
 
  if (window.widget) { // if in widget window execute flip
    setTimeout ('widget.performTransition();', 0);  
  }
  
  // draw the calendar list
  drawCalendarList();
}

function hidePrefs() {

  // save preferences and prepare to load front
  savePrefs();
  
  var front = document.getElementById("front");
  var back = document.getElementById("back");
   
  if (window.widget) {
    widget.prepareForTransition("ToFront");
  }
 
  back.style.display="none";
  front.style.display="block";
 
  if (window.widget) { // if in widget window execute flip
    setTimeout ('widget.performTransition();', 0);
  }

  // draw the events list
  setTimeout('refresh();', 0);
}

// Refresh --------------------------------------------------------------------

function refresh() {
  setTimeout('refresh();', kUpdateIntervalMs); // set widget to refresh at a constant frequency

  // clear the existing events
  todaysEvents = new Array();

  // get todays date
  var today = new Date();
  var isoDateToday = getDateIso8601(today);

  // determine if todays events show be shown
  var show = (widget.preferenceForKey('ShowTodaysEvents') == "true") ? 0 : 1;

  // set start time as today or today plus a day
  var startDate = new Date(today.getTime() + show * kOneDay);
  startDate.setHours(0);   // set time to 0
  startDate.setMinutes(0);
  startDate.setSeconds(0);
  startDate = toUTCTime(startDate);
  var startDay = getDateIso8601(startDate);

  // get days shown as days plus one if not showing today
  var days = widget.preferenceForKey('ShowDays');
  days = parseInt(days) + parseInt(show);

  // set end time as above
  var endDate = today;
  endDate.setDate(endDate.getDate() + days);
  endDate.setHours(0);   // set time to 0
  endDate.setMinutes(0);
  endDate.setSeconds(0);
  endDate = toUTCTime(endDate);
  var endDay = getDateIso8601(endDate);

  // iterate through calendars and get events for all
  for (var i = 0; i < calendars.length; i++) {
		if(document.getElementById(calendars[i].url).checked){ // only get events if the calendar is checked
			var thisUrl = calendars[i].url.replace(/basic$/g, 'full');
			addEventsFrom(thisUrl, startDay, endDay, kColors[i % kColors.length]);
		} // end if
  } // end for

 
}

// Get a Calendar Title ------------------------------------------------------------

function getCalendarTitle(url){
 // generate a request
 var xmlRequest = new XMLHttpRequest();

  // set the load properties and information to get
  xmlRequest.onload = function(e) { xmlLoadedCalendars(e, xmlRequest); }
  xmlRequest.setRequestHeader("Cache-Control", "no-cache");
  xmlRequest.overrideMimeType("application/atom+xml");
  
  try {
    // submit a request
    xmlRequest.open("GET", url, false);
    xmlRequest.send(null);
  }
  catch(e) {
    // check if the request could not be sent
    showError("Error sending HTTP request");
  }
}

function xmlLoadedCalendars(e, xmlRequest) {
  // if any status code is invalid return
  if (!xmlRequest || 
      xmlRequest.readyState != 4 || 
      xmlRequest.status != 200 || 
      !xmlRequest.responseXML) {
    return;
  }
  
  // find the first feed in the response
  var feed = findChild(xmlRequest.responseXML, 'feed');
  if (!feed) {
    alert("No <feed> element!");
    recurseAll(xmlRequest.responseXML, "");
    return;
  }

  // start at the begining element and search until title element is found
  for (entry = feed.firstChild; entry != null; entry = entry.nextSibling) {
    if (entry.nodeName == "title"){
	  newTitle = entry.firstChild.nodeValue; // set global newTitle to the calendar's title
	}
  }
  // reset the request
  xmlRequest = null;

}


// Add Events from Calendars -------------------------------------------------------

function addEventsFrom(url, expandFrom, expandTo, color) {
  // new request
  var xmlRequest = new XMLHttpRequest();

  // add start time and end time qualifiers to the feed request
  url = url + "?start-min=" + expandFrom
            + "&start-max=" + expandTo;

  // set to get feed elements on load
  xmlRequest.onload = function(e) { xmlLoadedEvents(e, xmlRequest, color); }
  xmlRequest.setRequestHeader("Cache-Control", "no-cache");
  xmlRequest.overrideMimeType("application/atom+xml");
  
  try {
    // request the feed
    xmlRequest.open("GET", url, false);
    xmlRequest.send(null);
  }
  catch(e) {
    // if request fails report error
    showError("Error sending HTTP request");
  }
}

function xmlLoadedEvents(e, xmlRequest, color) {
  // if request is not in ready state, return
  if (!xmlRequest || 
      xmlRequest.readyState != 4 || 
      xmlRequest.status != 200 || 
      !xmlRequest.responseXML) {
    return;
  }

  // get the first feed and check that it exists
  var feed = findChild(xmlRequest.responseXML, 'feed');
  if (!feed) {
    alert("No <feed> element!");
    recurseAll(xmlRequest.responseXML, "");
    return;
  }


  var entry, title, url, location, notes, reminder, attendees, startTime, endTime;

  // for each element check if it is entry    
  for (entry = feed.firstChild; entry != null; entry = entry.nextSibling) {	
	if (entry.nodeName != "entry") {		
      continue;
    }
    
    // for each entry get all the calendar entry data
    title = ""; url = ""; location = ""; notes = ""; reminder = ""; attendees = "";
    for (var node = entry.firstChild; node != null; node = node.nextSibling) {

      if (node.nodeName == "title")
        title = node.firstChild.nodeValue;
      if (node.nodeName == "link" && node.getAttribute("rel") == "alternate")
        url = node.getAttribute("href");
      if (node.nodeName == "gd:where")
        location = node.getAttribute("valueString");
      if (node.nodeName == "gd:reminder")
        reminder = node.getAttribute("minutes");
      if (node.nodeName == "gd:who") // Multiple gd:who
        attendees += node.getAttribute("valueString") + ", \n";
    }

    if (attendees.length > 3) { // Remove extra comma and space
      attendees = attendees.substr(0, attendees.length - 3);
    }

    for (var node = entry.firstChild; node != null; node = node.nextSibling) {
      // gd:whens will be multiple in case of recurrences.
      if (node.nodeName == "gd:when" && title != null) {
        var startTimeStr = node.getAttribute("startTime");
        var endTimeStr = node.getAttribute("endTime");


        isAllDay = (startTimeStr.length <= 11); // check if it is an all day event

        if (!isAllDay) {
		  // if not all day get date and time
          startTime = rfc3339StringToDate(startTimeStr); //parse iso8601 date
          endTime = rfc3339StringToDate(endTimeStr);
		  startTime = toLocalTime(startTime);  //convert to local time
		  endTime = toLocalTime(endTime);
        } else {
		  // if all day get date and set end time to correct day (by default it is one day to far)
          startTime = iso8601DateStringToDate(startTimeStr);
          startTime.setHours(0);   // set time to 0
          startTime.setMinutes(0);
          startTime.setSeconds(0);

		  endTime = iso8601DateStringToDate(endTimeStr);
		  endTime.setDate(endTime.getDate() - 1);
		  endTime.setHours(0);   // set time to 0
		  endTime.setMinutes(0);
		  endTime.setSeconds(0);
		  
        }
 
		var now = new Date();
        if ( isAllDay || 
             ( dateIsToday(startTime) && 
			   (widget.preferenceForKey("ShowPastEvents") == null || 
			   widget.preferenceForKey("ShowPastEvents") == "true") || 
			   startTime >= now )
           ) {
        	var calendarEvent = new CalendarEvent(title, url, startTime, endTime, 
              isAllDay, location, notes, reminder, attendees, color);
            todaysEvents.push(calendarEvent);
        } // end if should be shown
      } // end for each time occurance
    } // end for each event
  } // end for each entry

  if (todaysEvents.length == 0) {
    alert("No events retrieved.");
  }

  xmlRequest = null;

  // sort events by start time
  todaysEvents.sort(sortByDateFunc);
  // display events as they are added
  updateDisplay(todaysEvents);
}

function sortByDateFunc(a, b) { // Sorting function
  // return a - b
  return a.startTime.getTime() - b.startTime.getTime();
}

// Redraw front and back side dynamic lists ----------------------------------

function drawCalendarList() {
  // create a calendar list ans assign it to a div
  var calendarList = document.getElementById("calendarList");

  // remove previously exsisting calendars
  for (var i = calendarList.childNodes.length - 1; i >= 0; --i) {
    calendarList.removeChild(calendarList.lastChild);
  }
  
  // add new calendars elements and render them
  for (var i = 0; i < calendars.length; ++i) {
    calendarList.appendChild(calendars[i].render());
  }
}

function updateDisplay(events) {
  // create an events list and assign it to a div
  var eventList = document.getElementById("eventList");
 
  // remove previously existing events
  for (var i = eventList.childNodes.length - 1; i >= 0; --i) {
    eventList.removeChild(eventList.lastChild);
  }
  
  // add new events
  for (var i = 0; i < events.length; ++i) {
    eventList.appendChild(events[i].render());
  }
  
  // realign the scrollbar with the scroll area
  scrollArea.refresh();
}

// Mis ------------------------------------------------------------------------

function findChild (element, nodeName) {
  var child;
  for (child = element.firstChild; child != null; child = child.nextSibling)  {
    if (child.nodeName == nodeName) {
      return child;
    }
  }
  return null;
}

function recurseAll(element, spacing) {
  var child;
  for (child = element.firstChild; child != null; child = child.nextSibling)  {
    alert(spacing + child.nodeName + ":" + child.nodeValue);
    recurseAll(child, spacing + "  ");
  }
}

function changeCheck(url){ //Update a checkbox change while still on back
  for (var i = 0; i < calendars.length; i++){
    if(calendars[i].url == url){
		calendars[i].isActive = document.getElementById(url).checked;
	}
  }		
}

// CLASSES ////////////////////////////////////////////////////////////////

// Calendar Title with Checkbox -------------------------------------------

function Calendar(title, url, isActive){
  this.title = title;
  this.url = url;
  this.isActive = isActive;

  Calendar.prototype.render = Calendar_render;
  function Calendar_render() {
    // create proper table additions for a single row/calendar
	var calendar = document.createElement("tr");
		var checkCol = document.createElement("td");
			var checkbox = document.createElement("input");
				checkbox.setAttribute('type', 'checkbox');
				checkbox.setAttribute('id', this.url);
				checkbox.setAttribute('onClick', 'changeCheck(this.url);');
				checkbox.checked = this.isActive;
			checkCol.appendChild(checkbox);
		calendar.appendChild(checkCol);
		var titleCol = document.createElement("td");
			titleCol.innerText = this.title;
		calendar.appendChild(titleCol);
	return calendar;
  }
}

// Event entry with Title and color ----------------------------------------

function CalendarEvent(title, url, startTime, endTime, isAllDay, location,
    notes, reminder, attendees, color) {
  this.title = title;
  this.url = url;
  this.startTime = startTime;
  this.endTime = endTime;
  this.isAllDay = isAllDay;  
  this.location = location;
  this.notes = notes;
  this.reminder = reminder;
  this.attendees = attendees;
  this.color = color;

  CalendarEvent.prototype.render = CalendarEvent_render;
  function CalendarEvent_render() {
    var event = document.createElement('div');
    event.setAttribute('class', 'Event');
    if (this.url != null) {
	  // set attributes
      event.setAttribute ('eventTitle', this.title);
      event.setAttribute ('url', this.url);
      event.setAttribute ('startTime', this.startTime);
      event.setAttribute ('endTime', this.endTime);
      event.setAttribute ('isAllDay', this.isAllDay);
      event.setAttribute ('location', this.location);
      event.setAttribute ('notes', this.notes);
      event.setAttribute ('reminder', this.reminder);
      event.setAttribute ('attendees', this.attendees);

      event.setAttribute ('onclick', 'eventClick(this);');
      event.setAttribute ('onmouseover', 'highlightEvent(this);');
      event.setAttribute ('onmouseout', 'unhighlightEvent(this);');
    }
    
	// create the color block
    var colorBand = document.createElement('div');
    colorBand.style.backgroundColor = this.color;
    colorBand.style.height = '12px';
    colorBand.style.width = '11px';
    colorBand.style.cssFloat = 'left';
    colorBand.style.margin = '0px';
    colorBand.style.marginRight = '6px';
    colorBand.style.marginTop = '1px';
    colorBand.style.border = '1px solid #efefef';
    event.appendChild(colorBand);

    if (!this.isAllDay) {
	// if not all day give start time and day of week
      var timeSpan = document.createElement('span');
      timeSpan.setAttribute('class', 'Time');
      timeSpan.innerText = to12HourTime(this.startTime) + " " + getDayofWeek(this.startTime);
      event.appendChild(timeSpan);
    } 
	
	if (this.isAllDay) {
	// if all day configure days show properly
	  var timeSpan = document.createElement('span');
      timeSpan.setAttribute('class', 'Time');

	  var weekAhead = new Date();
	  weekAhead.setDate(weekAhead.getDate() + 7) // set up week from today constant
	  
	  if( weekAhead < this.endTime ){
	  // if ends more than one week from now, append "..."
	  	  var today = new Date();
		  if( dateIsToday(this.startTime) )//this.startTime.getDate() < today.getDate() )
		    // if starts today or before show today as start date, ignore past
			timeSpan.innerText = getDayofWeek(today) + "...";
		  else
		    // else show its start time
			timeSpan.innerText = getDayofWeek(this.startTime) + "...";
	  }
	  else if( getDayofWeek(this.startTime) == getDayofWeek(this.endTime) )
	    // if single day event show only the day
		timeSpan.innerText = getDayofWeek(this.startTime); 
	  else {
	    // otherwise show range
		var today = new Date();
		// same as above ignore the past
		if( dateIsToday(this.startTime) )//this.startTime.getDate() < today.getDate() )
			timeSpan.innerText = getDayofWeek(today) + "-" + getDayofWeek(this.endTime);
		else
			timeSpan.innerText = getDayofWeek(this.startTime) + "-" + getDayofWeek(this.endTime);
	  }
      event.appendChild(timeSpan);
	}
    
    var titleSpan = document.createElement('span');
    titleSpan.innerText = this.title;
    
	// set to bold if starts today or earlier
    if (dateIsToday(this.startTime)) {
      titleSpan.setAttribute('class', 'Title EventToday');
    } else {
      titleSpan.setAttribute('class', 'Title');
    }
    event.appendChild(titleSpan);


    
    return event;
  }
}

function eventClick(eventObject) {
// on click go to event
	if (window.widget) {
		widget.openURL (eventObject.url);
	} else {
	  document.location = eventObject.url;
	}
}

function highlightEvent(div) {
// set highlight status
	div.style.backgroundColor = "#e6e6e6";
	div.style.textShadow = "#b0b0b0 0px 2px 3px";
}

function unhighlightEvent(div) {
// reset highlight status
	div.style.backgroundColor = "";
	div.style.textShadow = "";
}

// HELPERS ////////////////////////////////////////////////////////////////////

function showHomePage() {
// show the product homepage
	if (window.widget) {
		widget.openURL(kProductHomePageUrl);
	} else {
	  document.location = kProductHomePageUrl;
	}
}

function onKeyPress() { // determine if the enter key is pressed
	if(event.keyCode == 13 || event.keyCode == 3) {
		addURL();
	}
}

// Date and Time functions/converters -------------------------------------------

function getDateIso8601(date) {
  // create a ISO8601 format date string of the given date type
  dateIso = date.getFullYear() + "-" + 
      (((date.getMonth() + 1) + "").length == 1 ? "0" + (date.getMonth() + 1) : (date.getMonth() + 1)) + "-" +
      ((date.getDate() + "").length == 1 ? "0" + date.getDate() : date.getDate()) + "T" +
	  ((date.getHours() + "").length == 1 ? "0" + date.getHours() : date.getHours()) + ":" +
	  ((date.getMinutes() + "").length == 1 ? "0" + date.getMinutes() : date.getMinutes()) + ":00";
  return dateIso;
}

function toLocalTime(utcDate) {  // utcDate is a Date date.
// adjust to computers local time, getTimezoneOffset is always computers local timezone regardless of the date given
  return new Date(utcDate.getTime() - (utcDate.getTimezoneOffset() * 1000 * 60) );
}

function toUTCTime(localDate) {  // localDate is a Date date.
// adjust computer's local time to UTC, getTimezoneOffset is always computers local timezone regardless of the date given
  return new Date(localDate.getTime() + (localDate.getTimezoneOffset() * 1000 * 60) );
}

function dateIsToday(date) {
  // check if the date is today or before today
  var today = new Date();
  if (date.getYear() <= today.getYear()
      && date.getMonth() <= today.getMonth()
      && date.getDate() <= today.getDate()) {
    return true;
  }
  return false;
}

function rfc3339StringToDate(rfc3339) { // returns date in UTC time
  // set a parse string to separate an ISO8601 date string
  var dateRegExp = /(\d\d\d\d)-(\d\d)-(\d\d)([Tt](\d\d):(\d\d):(\d\d)(\.(\d+))?)?([Zz]|((\+|-)(\d\d):(\d\d)))?/g;
  // divide the string
  var parts = dateRegExp.exec(rfc3339);

  if (parts.length != 15) {
    // Something weird happened with the Google feed again! :-(
    return null;
  }

  // set the elements of a date to the parts of the date string
  var d = new Date();
  d.setFullYear(parts[1]);
  d.setMonth(parseInt(parts[2]) - 1);
  d.setDate(parts[3]);
  d.setHours(parts[5]);
  d.setMinutes(parts[6]);
  d.setSeconds(parts[7]);

  // deal with timezone offset
  if(parts[13] != null && parts[14] != null) {
	// set a time zone offset for adjusting to local time
	var tzOffsetFeedMin = parts[13] * 60;
	if (parts[12] != "-") { // This is supposed to be backwards.
		tzOffsetFeedMin = -1 * tzOffsetFeedMin;	
	}
	d.setTime( d.getTime() + (tzOffsetFeedMin * 60 * 1000) ); // adjust to UTC if it is not there
  }
  
  return d;
}

function iso8601DateStringToDate(iso8601) {
  // same as above but disregard time, this is for all day events
  var dateRegExp = /(\d\d\d\d)-(\d\d)-(\d\d)/g;
  var parts = dateRegExp.exec(iso8601);

  if (parts.length != 4) {
    // Something weird happened with the Google feed again! :-(
    return null;
  }

  var d = new Date();
  d.setFullYear(parts[1]);
  d.setMonth(parseInt(parts[2]) - 1);
  d.setDate(parts[3]);

  return d;
}

function getDayofWeek(date) { 
// get 3 letter abbreviation for days
	switch( date.getDay() ){
	case 0:
		return "Sun";
	case 1:
		return "Mon";
	case 2:
		return "Tue";
	case 3:
		return "Wed";
	case 4:
		return "Thu";
	case 5:
		return "Fri";
	case 6:
		return "Sat";
	}
}

function to12HourTime(date) { 
// Convert 24hr Date format to 12hr string format
  var dateStr = "";

// adjust for 12 hr time format
  if (date.getHours() == "0") {
    dateStr += "12";
  } else if (date.getHours() > 0 && date.getHours() < 13) {
    dateStr += date.getHours();
  } else {
    dateStr += date.getHours() - 12;
  }

  
  if(date.getMinutes() != 0) {
  // if has minutes add them
	dateStr += ":";
	dateStr += (date.getMinutes() < 9) ? "0" + date.getMinutes()
                                     : date.getMinutes();
  }
  
  // add am or pm
  if (date.getHours() >= 12) {
    dateStr += "pm";
  } else {
    dateStr += "am";
  }

  return dateStr;
}